---------------------
---SAM THE MANAGER---
---------------------
--- This script was created with the purpose of managing large groups of SAMs more intelligently, without setting up dozens of triggers in the Mission Editor
--- MOOSE REQUIRED---
----------------------------
-- Utility: activate/deactivate groups with prefix
----------------------------
local function SetREDSAMGroups(prefix, state)
    for _, group in pairs(coalition.getGroups(coalition.side.RED, Group.Category.GROUND)) do
        if group and group:isExist() then
            local name = group:getName()
            if string.find(name, prefix) == 1 then
                local ctrl = group:getController()
                if ctrl then
                    ctrl:setOnOff(state) -- true = ON, false = OFF
                end
            end
        end
    end
end

----------------------------
-- Check if there are BLUE units (AIRPLANE/HELICOPTER) inside a zone
----------------------------
local function IsBlueInZone(zoneName)
    local zone = trigger.misc.getZone(zoneName)
    if not zone then return false end

    -- Merge AIRPLANE + HELICOPTER groups
    local allBlueGroups = {}
    for _, g in ipairs(coalition.getGroups(coalition.side.BLUE, Group.Category.AIRPLANE)) do
        table.insert(allBlueGroups, g)
    end
    for _, g in ipairs(coalition.getGroups(coalition.side.BLUE, Group.Category.HELICOPTER)) do
        table.insert(allBlueGroups, g)
    end

    -- Check all units
    for _, group in ipairs(allBlueGroups) do
        if group and group:isExist() then
            for _, unit in ipairs(group:getUnits()) do
                if unit and unit:isExist() then
                    local pos = unit:getPoint()
                    local dx = pos.x - zone.point.x
                    local dy = pos.z - zone.point.z
                    local distance = math.sqrt(dx * dx + dy * dy)
                    if distance <= zone.radius then
                        return true -- at least one unit found
                    end
                end
            end
        end
    end

    return false
end

----------------------------
-- SAM sector configuration
----------------------------
local SAM_Sectors_RED = {
    { prefix = "SAM_USSR_NORTH", zones = {"GDR-NORTH"} },
    { prefix = "SAM_USSR_WEST",  zones = {"GDR-WEST-1", "GDR-WEST-2", "GDR-WEST-3"} },
    { prefix = "SAM_USSR_EAST",  zones = {"GDR-EAST"} },
    { prefix = "SAM_USSR_SOUTH", zones = {"GDR-SOUTH"} }
}

----------------------------
-- Main control function
----------------------------
local function CheckAllSAMSectors()
    for _, sector in ipairs(SAM_Sectors_RED) do
        local active = false
        -- Check all zones of the sector
        for _, zoneName in ipairs(sector.zones) do
            if IsBlueInZone(zoneName) then
                active = true
                break
            end
        end

        -- Activate or deactivate SAMs in the sector
        SetREDSAMGroups(sector.prefix, active)
    end
end

----------------------------
-- Scheduler (every 10 seconds)
----------------------------
timer.scheduleFunction(
    function()
        CheckAllSAMSectors()
        return timer.getTime() + 10
    end,
    {},
    timer.getTime() + 1
)
